//
//  TodoList.swift
//  Do It
//
//  Created by Jim Dovey on 5/10/20.
//  Copyright © 2020 Pragmatic Programmers. All rights reserved.
//

import Foundation
import SwiftUI

struct TodoList: View {
    // START:ListDataCenter
    // START_HIGHLIGHT
    @EnvironmentObject private var data: DataCenter
    // END_HIGHLIGHT
    
    var list: TodoItemList? = nil
    // START_HIGHLIGHT
    var items: [TodoItem] {
        guard let list = list else { return data.todoItems }
        return data.items(in: list)
    }
    // END_HIGHLIGHT
    var title: String { list?.name ?? "All Items" }
    
    func color(for item: TodoItem) -> Color {
        // START_HIGHLIGHT
        let list = self.list ?? data.list(for: item)
        // END_HIGHLIGHT
        return list.color.uiColor
    }
    
    // <literal:elide> ... </literal:elide>
    // END:ListDataCenter
    @State private var sortBy: SortOption = .title
    @State private var showingChooser = false
    // START:ListDataCenter
    
    var body: some View {
        List(sortedItems) { item in
            NavigationLink(destination: TodoItemDetail(item: item)
                // START_HIGHLIGHT
                .environmentObject(self.data)
                // END_HIGHLIGHT
            ) {
                // <literal:elide> TodoItemRow(...) </literal:elide>
                // END:ListDataCenter
                TodoItemRow(item: item)
                    .accentColor(self.color(for: item))
                // START:ListDataCenter
            }
        }
        // <literal:elide> modifiers </literal:elide>
        // END:ListDataCenter
        .navigationBarTitle("To-Do Items")
        .navigationBarItems(trailing: sortButton)
        .listStyle(GroupedListStyle())
        .actionSheet(isPresented: $showingChooser) {
            ActionSheet(
                title: Text("Sort Order"),
                buttons: SortOption.allCases.map { opt in
                    ActionSheet.Button.default(Text(opt.rawValue)) {
                        self.sortBy = opt
                    }
            })
        }
        // START:ListDataCenter
    }
    // END:ListDataCenter
    
    private var sortButton: some View {
        Button(action: { self.showingChooser.toggle() }) {
            Image(systemName: "arrow.up.arrow.down.square")
                .imageScale(.large)
                .font(.system(size: 24, weight: .bold))
        }
    }
    
    private enum SortOption: String, CaseIterable {
        case title = "Title"
        case priority = "Priority"
        case dueDate = "Due Date"
    }
    
    private var sortedItems: [TodoItem] {
        items.sorted {
            switch sortBy {
            case .title:
                return $0.title
                    .caseInsensitiveCompare($1.title) == .orderedAscending
            case .priority:
                return $0.priority > $1.priority
            case .dueDate:
                return ($0.date ?? .distantFuture) < ($1.date ?? .distantFuture)
            }
        }
    }
}

// START:DataPreview
struct TodoList_Previews: PreviewProvider {
    static var previews: some View {
        NavigationView {
            TodoList()
        }
        // START_HIGHLIGHT
        .environmentObject(DataCenter())
        // END_HIGHLIGHT
    }
}
// END:DataPreview
